<?php
/**
 * Module Name: Ultimate Member for WordPress Integration
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( class_exists('UM') ) :

class WPAchievements_Module_UM {

  /**
   * Init required hooks
   *
   * @static
   * @access  public
   * @return  void
   */
  public static function init() {

    // Register settings hooks
    add_filter( 'wpachievements_get_sections_module', array( __CLASS__, 'register_section' ) );
    add_filter( 'wpachievements_achievements_modules_admin_settings', array( __CLASS__, 'admin_settings'), 10, 3 );

    // Register WPAchievements hooks
    add_action( 'wpachievements_admin_events', array( __CLASS__, 'trigger_dropdown' ) );
    add_filter( 'wpachievements_trigger_description', array( __CLASS__, 'selected_trigger' ) );
    add_filter( 'wpachievements_activity_description', array( __CLASS__, 'activity_description'), 10, 4 );
    add_filter( 'wpachievements_quest_description', array( __CLASS__, 'quest_description' ), 10, 3 );

    // Register Ultimate-Members hooks
    add_action( 'um_after_user_updated', array( __CLASS__, 'user_after_updating_profile' ), 10, 3 );
    add_action( 'um_after_user_account_updated', array( __CLASS__, 'user_after_updating_account' ), 10, 1 );
    add_action( 'um_on_login_before_redirect', array( __CLASS__, 'on_login_before_redirect' ), 10, 1 );
    add_action( 'um_after_remove_profile_photo', array( __CLASS__, 'after_remove_profile_photo' ) );
    add_action( 'um_after_remove_cover_photo', array( __CLASS__, 'after_remove_cover_photo' ) );
    add_action( 'um_after_user_is_approved', array( __CLASS__, 'after_user_is_approved' ) );
    add_action( 'um_after_user_is_inactive', array( __CLASS__, 'after_user_is_inactive' ) );
    add_action( 'um_after_upload_db_meta', array( __CLASS__, 'after_upload_db_meta' ), 10, 2 );

    // Additional Functions
    add_filter( 'um_user_profile_tabs', array( __CLASS__, 'extend_um_user_profile_tabs' ) );
    add_action( 'um_profile_content_wpachievements', array( __CLASS__, 'um_profile_content_wpachievements' ) );
    add_action( 'um_after_profile_header_name', array( __CLASS__, 'um_profile' ) );
    add_action( 'um_members_after_user_name', array( __CLASS__, 'um_members_grid' ) );
  }

  /**
  * Triggered on user login
  *
  * @static
  * @access  public
  * @param   int $um_user_id   UM User ID
  * @return  void
  */
  public static function on_login_before_redirect( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_on_login',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_on_login'),
    ) );
  }

  /**
   * Triggered on user profile update
   *
   * @static
   * @access  public
   * @param   int       $um_user_id   User ID
   * @param   array     $args         Form Data
   * @param   array     $submitted    Submitted User Data  
   * @return  void
   */
  public static function user_after_updating_profile( $um_user_id, $args, $submitted ){
    if( get_current_user_id() !== $um_user_id ) {
        return;
    }
    if( isset( $submitted['description'] ) ) {
      WPAchievements_Trigger::new_activity( array(
        'activity'    => 'um_updating_profile',
        'user_id'     => $um_user_id,
        'post_id'     => '',
        'points'      => (int) wpachievements_get_site_option('wpachievements_um_updating_profile'),
      ) );
    }
  }

  /**
   * Triggered on user account update
   *
   * @static
   * @access  public
   * @param   int $um_user_id   UM User ID
   * @return  void
   */
  public static function user_after_updating_account( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_updating_account',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_updating_account'),
    ) );    
  }

  /**
   * Triggered on user is removing profile photo
   *
   * @static
   * @access  public
   * @param   int $um_user_id   UM User ID
   * @return  void
   */
  public static function after_remove_profile_photo( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_after_remove_profile_photo',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_remove_profile_photo'),
    ) );    
  }

  /**
   * Triggered on user is removing cover photo
   *
   * @static
   * @access  public
   * @param   int $um_user_id   UM User ID
   * @return  void
   */
  public static function after_remove_cover_photo( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_after_remove_cover_photo',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_remove_cover_photo'),
    ) );    
  }

  /**
   * Triggered after user is approved
   *
   * @static
   * @access  public
   * @param   int $um_user_id   UM User ID
   * @return  void
   */
  public static function after_user_is_approved( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_after_user_is_approved',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_user_is_approved'),
    ) );    
  }

  /**
   * Triggered after user is inactive
   *
   * @static
   * @access  public
   * @param   int $um_user_id   UM User ID
   * @return  void
   */
  public static function after_user_is_inactive( $um_user_id ){
    WPAchievements_Trigger::new_activity( array(
      'activity'    => 'um_after_user_is_inactive',
      'user_id'     => $um_user_id,
      'post_id'     => '',
      'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_user_is_inactive'),
    ) );    
  }

  /**
   * Triggered after user uploaded a profile or cover photo
   *
   * @static
   * @access  public
   * @param   int     $um_user_id   UM User ID
   * @param   string  $key          Meta key
   * @return  void
   */
  public static function after_upload_db_meta( $um_user_id, $key ){
    if ( 'profile_photo' == $key ) {
      WPAchievements_Trigger::new_activity( array(
        'activity'    => 'um_after_upload_profile_photo',
        'user_id'     => $um_user_id,
        'post_id'     => '',
        'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_upload_profile_photo'),
      ) );    
    }
    if ( 'cover_photo' == $key ) {
      WPAchievements_Trigger::new_activity( array(
        'activity'    => 'um_after_upload_cover_photo',
        'user_id'     => $um_user_id,
        'post_id'     => '',
        'points'      => (int) wpachievements_get_site_option('wpachievements_um_after_upload_cover_photo'),
      ) );    
    }
  }

  /**
  * Register Extend_um_user_profile_tabs
  *
  * @static
  * @access  public
  * @param   array $tabs Array of menu tabs in the profile view of the user
  * @return  array Tabs
  */
  public static function extend_um_user_profile_tabs( $tabs ) {
    if ( 'Disable' != wpachievements_get_site_option('wpachievements_um_show_achievements_tab') || 'Disable' != wpachievements_get_site_option('wpachievements_um_show_quests_tab') ) {
      $tabs['wpachievements'] = array(  "name" => "Achievements", "icon" => "um-faicon-trophy" );
    }
    return $tabs;
  }

  /**
  * Register Extend_um_user_profile_content
  *
  * @static
  * @access  public
  * @param   none
  * @return
  */
  public static function um_profile_content_wpachievements() {
    if ( 'Disable' != wpachievements_get_site_option('wpachievements_um_show_achievements_tab') || 'Disable' != wpachievements_get_site_option('wpachievements_um_show_quests_tab') ) {

      // Get Bages max number.
      $max_badges = intval( wpachievements_get_site_option( 'wpachievements_um_max_show_tab_badges' ) );
      $limit = -1;
      if ($max_badges > 0) {
        $limit = $max_badges;
      }
      // Get Bages size.
      $size_badges = intval( wpachievements_get_site_option( 'wpachievements_um_size_show_tab_badges' ) );

      if ( !$size_badges || $size_badges < 0) { $size_badges = 50; }
   
      echo '<div class="um-ajax-items um-profile-note">';
      if ( 'Disable' != wpachievements_get_site_option('wpachievements_um_show_achievements_tab') ) {
        echo do_shortcode('[wpa_myachievements user_id="'.um_profile_id().'" show_title="true" title_heading="h5" image_width="'.$size_badges.'" achievement_limit="'.$limit.'"]' );
      }
      if ( 'Disable' != wpachievements_get_site_option('wpachievements_um_show_quests_tab') ) {
         echo do_shortcode('[wpa_myquests user_id="'.um_profile_id().'" show_title="true" title_heading="h5" image_width="'.$size_badges.'" achievement_limit="'.$limit.'"]' );
      }
      echo '<br /><div class="clear"></div></div>';
    }
  }

  /**
  * Show rank stats in profile
  *
  * @static
  * @access  public
  * @param   int $displayed_userid User ID of the profile
  * @return
  */
  public static function um_profile(){
    if ( 'Disable' == wpachievements_get_site_option('wpachievements_rank_status') ) {
      return;
    }
    echo '<div class="um-member-meta-wpachievements"><strong>'.__('Rank', 'wpachievements').':</strong> '.wpachievements_getRank( um_profile_id() ).'</div>';
  }

  /**
  * Show rank status in profile member grid
  *
  * @static
  * @access  public
  * @param   int $displayed_userid User ID of the profile
  * @return
  */
  public static function um_members_grid( $displayed_userid ){
    if ( 'Disable' != wpachievements_get_site_option('wpachievements_rank_status') ) {
      echo '<div class="um-member-name"><strong>'.__('Rank', 'wpachievements').':</strong></br>'.wpachievements_getRank($displayed_userid).'</div>';
    }

    // Get badges visibility
    if ( 'Disable' == wpachievements_get_site_option('wpachievements_um_show_achquests_user_card') ) {
      return;
    }

    // Get Bages max number.
    $max_badges = intval( wpachievements_get_site_option( 'wpachievements_um_max_card_user_badges' ) );
    $limit = -1;
    if ($max_badges > 0) {
      $limit = $max_badges;
    }
    // Get Bages size.
    $size_badges = intval( wpachievements_get_site_option( 'wpachievements_um_size_card_user_badges' ) );
    if ( !$size_badges || $size_badges < 0) { $size_badges = 30; }
    
    echo '<div class="um-member-name"><strong>'.__('Achievements', 'wpachievements').':</strong>';
    echo do_shortcode('[wpa_myachievements user_id="'.$displayed_userid.'" show_title="false" image_width="'.$size_badges.'" achievement_limit="'.$limit.'"]' );
    echo '</div>';
    echo '<div class="um-member-name"><strong>'.__('Quests', 'wpachievements').':</strong>';
    echo do_shortcode('[wpa_myquests user_id="'.$displayed_userid.'" show_title="false" image_width="'.$size_badges.'" achievement_limit="'.$limit.'"]' );
    echo '</div>';
  }
  /**
   * Register admin section
   *
   * @static
   * @access  public
   * @param   array $sections Array of registered sections for admin settings
   * @return  array           Sections
   */
  public static function register_section( $sections ) {
    $sections['um'] = __("Ultimate-Member", 'wpachievements' );

    return $sections;
  }

  /**
   * Admin settings
   *
   * @static
   * @access  public
   * @param   array $default_settings
   * @param   string $short_name
   * @param   string $section
   * @return  array
   */
  public static function admin_settings( $default_settings, $short_name, $section ) {

    if ( $section !== 'um' ) {
      return $default_settings;
    }

    $settings = array();
    $settings[] = array( 'title' => __( 'Ultimate-Member', 'wpachievements' ), 'type' => 'title', 'desc' => '', 'id' => 'UM_options' );

    $settings[] = array(
      'title'   => __( 'User profile update', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user updates his profile.', 'wpachievements' ),
      'id'      => $short_name.'_um_updating_profile',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User account update', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user updates his account.', 'wpachievements' ),
      'id'      => $short_name.'_um_updating_account',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User logs in', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user is logging in.', 'wpachievements' ),
      'id'      => $short_name.'_um_on_login',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User remove profile photo', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user removes the profile photo of his account.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_remove_profile_photo',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User remove cover photo', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user removes the cover photo of his account.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_remove_cover_photo',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User profile approved', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user profile is approved.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_user_is_approved',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User profile inactive', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user profile is inactive.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_user_is_inactive',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User uploads a profile photo', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user uploaded a profile photo.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_upload_profile_photo',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'User uploads a cover photo', 'wpachievements' ),
      'desc'    => __( 'Points awarded when a user uploaded a cover photo.', 'wpachievements' ),
      'id'      => $short_name.'_um_after_upload_cover_photo',
      'type'    => 'number',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'Achievements Tab Content', 'wpachievements' ),
      'desc'    => __( 'Select to show Achievements in the user tab.', 'wpachievements' ),
      'id'      => 'wpachievements_um_show_achievements_tab',
      'type'    => 'select',
      'default' => 'Enable',
      'css'     => 'min-width:300px;',
      'options' => array(
        'Enable'  => __('Enable','wpachievements' ),
        'Disable'  => __('Disable', 'wpachievements' ),
      ),
    );

    $settings[] = array(
      'title'   => __( 'Quests Tab Content', 'wpachievements' ),
      'desc'    => __( 'Select to show Quests in the user tab.', 'wpachievements' ),
      'id'      => 'wpachievements_um_show_quests_tab',
      'type'    => 'select',
      'default' => 'Enable',
      'css'     => 'min-width:300px;',
      'options' => array(
        'Enable'  => __('Enable','wpachievements' ),
        'Disable'  => __('Disable', 'wpachievements' ),
      ),
    );

    $settings[] = array(
      'title'   => __( 'Max. Tab Content Items', 'wpachievements' ),
      'desc'    => __( 'Enter the number of max items to show.<br/>(Enter 0 to show all)', 'wpachievements' ),
      'id'      => 'wpachievements_um_max_show_tab_badges',
      'type'    => 'text',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'Size Tab Content Badges', 'wpachievements' ),
      'desc'    => __( 'Enter the size of badges in px.<br/>(Default: 50)', 'wpachievements' ),
      'id'      => 'wpachievements_um_size_show_tab_badges',
      'type'    => 'text',
      'default' => '50',
    );

    $settings[] = array(
      'title'   => __( 'User/Member Card Grid', 'wpachievements' ),
      'desc'    => __( 'Select to show Achievments/Quests in the user grid card.', 'wpachievements' ),
      'id'      => 'wpachievements_um_show_achquests_user_card',
      'type'    => 'select',
      'default' => 'Enable',
      'css'     => 'min-width:300px;',
      'options' => array(
        'Enable'  => __('Enable','wpachievements' ),
        'Disable'  => __('Disable', 'wpachievements' ),
      ),
    );

    $settings[] = array(
      'title'   => __( 'Max. User/Member Card Items', 'wpachievements' ),
      'desc'    => __( 'Enter the number of max items to show.<br/>(Enter 0 to show all)', 'wpachievements' ),
      'id'      => 'wpachievements_um_max_card_user_badges',
      'type'    => 'text',
      'default' => '0',
    );

    $settings[] = array(
      'title'   => __( 'Size User/Member Card Grid Badges', 'wpachievements' ),
      'desc'    => __( 'Enter the size of badges in px.<br/>(Default: 30)', 'wpachievements' ),
      'id'      => 'wpachievements_um_size_card_user_badges',
      'type'    => 'text',
      'default' => '30',
    );

    $settings[] = array( 'type' => 'sectionend', 'id' => 'UM_options');

    return $settings;
  }

  /**
   * Return all available triggers with descriptions
   *
   * @static
   * @access  public
   * @return  array Array of triggers and descriptions
   */
  public static function get_triggers() {
    return array(
      "um_updating_profile" => __('The user updates his profile', 'wpachievements'),
      "um_updating_account" => __('The user updates his account', 'wpachievements'),
      "um_on_login" => __('The user is logging in', 'wpachievements'),
      "um_after_remove_profile_photo" => __('The user is removing the profile photo', 'wpachievements'),
      "um_after_remove_cover_photo" => __('The user is removing the cover photo', 'wpachievements'),
      "um_after_user_is_approved" => __('The user is approved', 'wpachievements'),
      "um_after_user_is_inactive" => __('The user is inactive', 'wpachievements'),
      "um_after_upload_profile_photo" => __('The user is uploading a profile photo', 'wpachievements'),
      "um_after_upload_cover_photo" => __('The user is uploading a cover photo', 'wpachievements'),
    );
  }

  /**
   * Register trigger events displayed on the achievements page
   *
   * @static
   * @access  public
   * @return  string
   */
  public static function trigger_dropdown() {
    ob_start();
    ?>
    <optgroup label="Ultimate-Member Events">
      <?php foreach ( self::get_triggers() as $key => $description ) : ?>
        <option value="<?php echo $key; ?>"><?php echo $description; ?></option>
      <?php endforeach; ?>
    </optgroup>
    <?php
    ob_end_flush();
  }

  /**
   * Return description of selected trigger
   *
   * @static
   * @access  public
   * @param   string $trigger Trigger key
   * @return  string          Trigger description
   */
  public static function selected_trigger( $trigger ) {
    $triggers = self::get_triggers();

    if ( array_key_exists( $trigger, $triggers ) ) {
      return $triggers[ $trigger ];
    }

    return $trigger;
  }

  /**
   * Generate activity descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Activity description
   * @param   string $type  Activity trigger
   * @param   int $points   Points
   * @param   int $times    Occurrences
   * @return  string        Activity description
   */
  public static function activity_description( $text, $type, $points, $times ) {

    switch( $type ) {
      case 'um_updating_profile': {
        $text = sprintf( __( 'for updating %s profile', 'wpachievements' ), $times );
      } break;

      case 'um_updating_account': {
        $text = sprintf( __( 'for updating %s account', 'wpachievements' ), $times );
      } break;

      case 'um_on_login': {
          $text = sprintf( __( 'for %s logging in', 'wpachievements' ), $times );
      } break;

      case 'um_after_remove_profile_photo': {
          $text = sprintf( __( 'for %s removing a profile photo', 'wpachievements' ), $times );
      } break;

      case 'um_after_remove_cover_photo': {
          $text = sprintf( __( 'for %s removing a cover photo', 'wpachievements' ), $times );
      } break;

      case 'um_after_user_is_approved': {
          $text = sprintf( __( 'for %s being approved', 'wpachievements' ), $times );
      } break;   

      case 'um_after_user_is_inactive': {
          $text = sprintf( __( 'for %s being inactive', 'wpachievements' ), $times );
      } break;

      case 'um_after_upload_profile_photo': {
          $text = sprintf( __( 'for %s uploading a profile photo', 'wpachievements' ), $times );
      } break;

      case 'um_after_upload_cover_photo': {
          $text = sprintf( __( 'for %s uploading a cover photo', 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }

  /**
   * Generate quest step descriptions
   *
   * @static
   * @access  public
   * @param   string $text  Step description
   * @param   string $type  Trigger type
   * @param   int $times    Occurrences
   * @return  string        Step description
   */
  public static function quest_description( $text, $type, $times ) {

    switch( $type ) {
      case 'um_updating_profile': {
        $text = sprintf( __( 'Updated %s profile', 'wpachievements' ), $times );
      } break;

      case 'um_updating_account': {
        $text = sprintf( __( 'Updated %s account', 'wpachievements' ), $times );
      } break;

      case 'um_on_login': {
        $text = sprintf( __( 'The user is logging in %s', 'wpachievements' ), $times );
      } break;

      case 'um_after_remove_profile_photo': {
          $text = sprintf( __( 'Removed %s profile photo', 'wpachievements' ), $times );
      } break;

      case 'um_after_remove_cover_photo': {
          $text = sprintf( __( 'Removed %s cover photo', 'wpachievements' ), $times );
      } break;

      case 'um_after_user_is_approved': {
          $text = sprintf( __( 'Approved %s user accounts', 'wpachievements' ), $times );
      } break;   

      case 'um_after_user_is_inactive': {
          $text = sprintf( __( 'Inactive %s user accounts', 'wpachievements' ), $times );
      } break;

      case 'um_after_upload_profile_photo': {
          $text = sprintf( __( 'Uploaded %s profile photo', 'wpachievements' ), $times );
      } break;
      
      case 'um_after_upload_cover_photo': {
          $text = sprintf( __( 'Uploaded %s cover photo', 'wpachievements' ), $times );
      } break;
    }

    return $text;
  }
}

WPAchievements_Module_UM::init();

endif;